; the code to analyze the cnf files we are generating.

(defun in (a X)
  (cond ((endp X) nil)
        ((equal a (car X)) t)
        (t (in a (cdr X)))))

(defun add (x Y)
"Adds x to the set Y"
  (if (in x Y)
      Y
    (cons x Y)))

(defun set-union (X Y)
  (if (endp X)
      Y
    (add (car X) (set-union (cdr X) Y))))

(defun << (x y)
  "Used to sort clauses.  We want literals & their negations next
to one another."
  (let ((ax (abs x))
        (ay (abs y)))
    (< ax ay)))

(defun sz (l)
  "The size of a cnf formula is the number of literals in it."
  (reduce #'+ (mapcar #'length l)))

(defun simp (l)
  "The first simplification.  We assume l is a clause and remove
duplicate literals and also return nil (empty clause, or true) if
a literal & its negation appear."
  (cond ((endp l) l)
        ((endp (cdr l)) l)
        ((equal (first l) (second l))
         (simp (cdr l)))
        ((equal (abs (first l))
                (abs (second l)))
         nil)
        (t (cons (car l) (simp (cdr l))))))

(defun simp2-a (l ll)
  (cond ((endp l)
         (reverse ll))
        ((null (car l))
         (simp2-a (cdr l) ll))
        (t (simp2-a (cdr l)
                    (cons (car l) ll)))))
         
(defun simp2 (l)
  "The second simplification.  We assume l is a list of clauses
& we remove any empty clauses."
  (simp2-a l nil))

(defun simp3 (l)
  "The third simplification. We assume l is a list of clauses and
check if the first 2 clauses contain 2 literals and can be
resolved away."
  (cond ((endp l) nil)
        ((endp (cdr l)) nil)
        (t (let ((l1 (car l))
                 (l2 (second l)))
             (if (and (equal 2 (length l1))
                      (equal 2 (length l2))
                      (equal (car l1) (- (car l2)))
                      (equal (second l1) (second l2)))
                 (cons (list (second l1))
                       (simp3 (cddr l)))
               l)))))

(defun extract-units-a (l ll)
  (cond ((endp l)
         ll)
        ((equal (length (car l)) 1)
         (extract-units-a (cdr l) (add (caar l) ll)))
        (t (extract-units-a (cdr l) ll))))

(defun extract-units (l)
  (extract-units-a l nil))

(defun simp4 (l u)
  "The fourth simplification.  We assume that l is a clause and u
a list of literals.  We simplify assuming all literals in u are
true."
  (if (endp l)
      nil
    (if (in (- (car l)) u)
        (simp4 (cdr l) u)
      (if (in (car l) u)
          nil
        (cons (car l) (simp4 (cdr l) u))))))

(defun pcnf (l)
  "Given a list of clauses, print the corresponding cnf file, as
a string"
  (if (endp l)
      (format nil "0~%")
    (format nil "~A ~A" (car l) (pcnf (cdr l)))))


(progn (setf cnf1 (copy-seq *c1*)) nil)
(progn (setf cnf2 (mapcar (lambda (x) (sort x #'<<)) cnf1)) nil)
(progn (setf cnf3 (mapcar #'simp cnf2)) nil)
(progn (setf cnf4 (simp2 cnf3)) nil)
(progn (setf cnf5 (simp3 cnf4)) nil)
(progn (setf u (extract-units cnf5)) nil)
(progn (setf cnf6 (simp2 (mapcar (lambda (x) (simp4 x u)) cnf5)))
       nil)

(setf u (extract-units cnf6))
(progn (setf cnf7 (simp2 (mapcar (lambda (x) (simp4 x u)) cnf6)))
       nil)
(setf u (extract-units cnf7))
(progn (setf cnf8 (simp2 (mapcar (lambda (x) (simp4 x u)) cnf7)))
       nil)
(setf u (extract-units cnf8))
(progn (setf cnf9 (simp2 (mapcar (lambda (x) (simp4 x u)) cnf8)))
       nil)
(setf u (extract-units cnf9))
(progn (setf cnf10 (simp2 (mapcar (lambda (x) (simp4 x u)) cnf9)))
       nil)



(defun print-cnf (l file)
  (let ((f (format nil "~a.cnf" file)))
    (with-open-file (out f
                         :direction :output
                         :if-exists :supersede
                         :if-does-not-exist :create)
      (dolist (c l)
        (format out "~a" (pcnf c))))))


; (print-cnf '((1 2 3) (-2 1 9)) "foo")

(print-cnf cnf6 "foo")

(defun gen-cnf (l)
  "We assume l is a list of clauses & we simplify as per above."
  (let* ((cnf1 (copy-seq l))
         (cnf2 (mapcar (lambda (x) (sort x #'<<)) cnf1))
         (cnf3 (mapcar #'simp cnf2))
         (cnf4 (simp2 cnf3))
         (cnf5 (simp3 cnf4))
         (u (extract-units cnf5))
         (cnf6 (simp2 (mapcar (lambda (x) (simp4 x u)) cnf5))))
    (format t "~A"
            (reduce (lambda (x y) (concatenate 'string x y))
                    (mapcar #'pcnf cnf6)))
    cnf6))


; (gen-cnf *cnf*)
; (reduce #'set-union (mapcar (lambda (l) (mapcar #'abs l)) cnf6)) 
; so there are only 8 variables here.

; here is what seems to be happening: for nd,md,ni set to n &
; nr,os set to 2,6 we have a truth table for some number of vars:
;
;
; n=2: 5 vars (44-48)
; n=3: 6 vars (91-96)
; n=4: 7 vars (186-192)
; n=5: 8 vars (377-384)
; n=6: 9 vars (760-768)
;
; 
; n=x: x+3 vars (2f(x-1)+x---2f(x-1)+x+x+2)
; f(2)=44

(defun fs (n)
  "The first var for n, see above"
  (cond ((< n 2) 0)
        ((= n 2) 44)
        (t (+ (* 2 (fs (1- n))) n))))

(defun fn (n)
  "The last var for n, see above"
  (+ (fs n) n 2))

; (mapcar (lambda (x) (cons (fs x) (fn x))) '(2 3 4 5 6 7 8))
